/* ------------------------------------------------------------------------------------*/
/*                          Bootstrap Class,  Version 3                                */
/*                          requires Ox 3.x 							       	       */
/*                          James Davidson  (james.davidson@exeter.ac.uk)              */
/*                          Release of 14-08-2005                                      */

/*  This file may be distributed and altered freely, but please retain this header.
	 In case of modifications, kindly email an annotated copy to the author.     	   */
/* ------------------------------------------------------------------------------------*/

#include <oxstd.h>
#include <oxfloat.h>

class Bootstrap
{
	decl m_HFileIn;                       
	decl m_bDblBoot;		                         
	decl m_bFstDblBoot;		                             
	decl m_cReps;	 		                
	decl m_cDblReps;		
	decl m_dAlphaMax;		
	decl m_bCorrectedPval;	  
	decl m_mrX; 			  
	decl m_OutFile;
	decl m_bFile;
	decl m_mStats;
	decl m_vrOrds;
	decl m_aAlpha;
	decl m_bConfInt;
	decl m_aNames;
	decl m_dP;
	decl m_dQ;
	decl m_dVP;
	decl m_dVQ;
	Bootstrap();
	virtual StoreParams();
	virtual RetrieveParams(const aStore);
	virtual Estimate(const mcData);
	virtual BootstrapSample(const mcResids);
	virtual LoadModel(const mcData);
	virtual PrintCall(const bLine, ...);
	Defaults();
	Currenttime();
	PvalCorrect(const mPval, const mPvalfix);
	PrintHeading();
	Shocks(const mrResids, const cExtra);
	BlockShocks(const mrResids, const cExtra, const bLength);
	WildShocks(const mrResids, const bType);
	PeeValues(const vActuals, const mStats);
	ReturnDist();
	Quantiles(const vPhats, const mStats);
	DoubleBoot(const mStats, const iNdices, const dPeeval);
	Test(const mrResids, const vActuals);
	Settings(const aArgs0, ...);
	TestCall(const mcData);
};

Bootstrap::Bootstrap()
{
	if (oxversion() < 300)
        PrintCall(1,"Requires Ox 3.0 or newer!\n");
		Defaults();
	m_vrOrds = <.01,0.025,0.05,0.1,0.2,0.3,0.5,0.7,0.8,0.9,0.95,0.975,0.99>;
	m_aAlpha = {"  0.01:"," 0.025:","  0.05:","   0.1:","   0.2:","   0.3:",
				"   0.5:","   0.7:","   0.8:","   0.9:","  0.95:"," 0.975:",
						"  0.99:"};
	m_dP = (sqrt(5)	+ 1)/(2*sqrt(5));
	m_dVP = -(sqrt(5) - 1)/2;
	m_dVQ = (sqrt(5) + 1)/2;
}

Bootstrap::StoreParams() {}
Bootstrap::RetrieveParams(const aStore) {}

Bootstrap::PrintCall(const bLine, ...)
{
	decl args = va_arglist();
	if(m_bFile) decl outfile = fopen(m_OutFile, "a"); 
	for (decl i = 0; i<sizeof(args); i++)
	{
		print(args[i]);
		if(m_bFile) fprint(outfile, args[i]);
	}
	if (bLine)
	{
		println("");
		if(m_bFile) fprintln(outfile, "");
	}
	if(m_bFile) fclose(outfile);
}

Bootstrap::Defaults()
{
	m_bFstDblBoot = m_bDblBoot = 0;
	m_cDblReps = m_cReps = 399;
	m_OutFile = m_HFileIn = "";
	m_bFile = 0;
	ranseed(Currenttime());
	m_dAlphaMax = 1;
	m_bCorrectedPval = 0;
	m_bConfInt = 0;
	m_aNames = {"Statistic"};
}

Bootstrap::Currenttime()
{
 decl h, m, s, stim = time();
 sscan(stim[0:1], "%d", &h);
 sscan(stim[3:4], "%d", &m);
 sscan(stim[6:7], "%d", &s);
 return h * 3600 + m * 60 + s;
}

Bootstrap::PvalCorrect(const mPval, const mPvalfix)
{
	decl cstats = rows(mPval);
	decl cpvals = columns(mPval);
	decl mcorpv = zeros(mPval);
	decl apv = new array[cstats];
	for (decl ac = 0; ac < cstats; ac++)
		apv[ac] = mPvalfix[][ac*cpvals:(ac+1)*cpvals-1];
	decl cbins = rows(mPvalfix);
	for (decl ac = 0; ac < cstats; ac++) 
	for (decl j = 0; j < cpvals; j++)
		mcorpv[ac][j] = (apv[ac])[ceil(mPval[ac][j]*(cbins-1))][j];
	return mcorpv;
}	

Bootstrap::Shocks(const mcResids, const cExtra)
{
	decl cc = rows(mcResids);
	decl ccl = cc+cExtra;
	decl mcshocks = zeros(ccl, columns(mcResids));
 	decl munidraw = floor(ranu(1,ccl)*cc);
	for (decl i=0; i<ccl; i++)
		mcshocks[i][] = mcResids[munidraw[i]][];
	return mcshocks;
}

Bootstrap::BlockShocks(const mcResids, const cExtra, const cBlength)
{
	if (cBlength <= 1) return Shocks(mcResids, cExtra);
	decl cc = rows(mcResids);
	decl ccp = cc + cExtra;
	decl ccl = cc + cExtra - cBlength;
	decl cblocks = ceil(cc/cBlength);
	decl mcshocks = <>; //zeros(ccl, columns(mcResids));
 	decl munidraw = floor(ranu(1,cblocks)*ccl);
	for (decl i=0; i<cblocks; i++)
		mcshocks |= mcResids[munidraw[i]: munidraw[i] + cBlength - 1][];
	return mcshocks[:ccp-1][];
}

Bootstrap::WildShocks(const mcResids, const iType)
{
 	decl munidraw = ranu(rows(mcResids), 1);
	if (iType == 0)
		munidraw = (munidraw .> 0.5) .? 1 .: -1;
	else if (iType == 1)
		munidraw = (munidraw .< m_dP) .? m_dVP .:  m_dVQ;
	return  mcResids .* munidraw;
}

Bootstrap::PeeValues(const vcActual, const mStats)
{
	decl cases =  sizer(vcActual);
	decl creps = columns(mStats);
	decl pval = zeros(cases,1);
	for (decl j = 0; j < cases; j++) 
	for (decl i = 0; i < creps; i++)
		if (vcActual[j] <= mStats[j][i])
		{
			pval[j] = 1-i/creps;
			break;
		}
	return pval;
}

Bootstrap::ReturnDist()
{
	return m_mStats;
}

Bootstrap::Quantiles(const vcOrd, const mStats)
{
	decl cases = rows(mStats);
	decl vcquantile = zeros(cases,1);
	for (decl j = 0; j < cases; j++)
		vcquantile[j] = mStats[j][min(max(0, floor(vcOrd[j])),
											columns(mStats)-1)];	
	return vcquantile;
}

Bootstrap::DoubleBoot(const mStats, const iNdex, const dPeeval)
{
	decl ds1, ds2 = 0;
	decl count;
	decl ctests = rows(mStats); 
	decl mpv = dPeeval*m_cDblReps;
	decl dblresids, i1, j1;
	decl mdblpeeval = ones(ctests, m_cReps);
	decl mdblstats;
	for (decl i=0; i<m_cReps; i++)							   
	{
		count = i;
		dblresids = LoadModel((m_mrX[iNdex[i]])');
		ds1 = 0;
		for (decl j=0; j<m_cDblReps; j++)
		{
			mdblstats =	(Estimate(BootstrapSample(dblresids)))';
			if (mdblstats >= mStats[i]) ds1 += 1;
			if (ds1 > mpv) break;
			if (ds1 + m_cDblReps - j - 1 <= mpv)
			{
				ds2 += 1;
				break;
			}	
		}
		if (ds2 > m_cReps*m_dAlphaMax) break; 
	}
	return ds2/m_cReps;
}	

Bootstrap::Test(const mcResids, const vcActual)
{
	m_mrX = new array [m_cReps];
	decl mstats = zeros(rows(vcActual), m_cReps), mstatsfd;
	decl storedpar, mcx, pv;
	decl ds = 0;
	if (m_bFstDblBoot)
	{	storedpar = StoreParams();
		mstatsfd = mstats;
	}
	for (decl i=0; i<m_cReps; i++)
	{
		mcx = BootstrapSample(mcResids);
		mstats[][i] = Estimate(mcx);
		if (m_bDblBoot) m_mrX[i] = mcx';
		if (m_bFstDblBoot)
		{
			mcx = BootstrapSample(LoadModel(mcx));
			mstatsfd[][i] = Estimate(mcx);
			RetrieveParams(storedpar);
		}
	}
	decl index = sortcindex(mstats[0][]);
	m_mStats = sortr(mstats);
	if (m_bConfInt)
	{
		decl critval = zeros(sizer(m_mStats),	sizec(m_vrOrds));
		for (decl j=0; j<sizec(m_vrOrds); j++)
			critval[][j] = Quantiles(ones(sizer(m_mStats),1)
								*m_vrOrds[j]*m_cReps, m_mStats);
		return critval;	
	}
	else
	{
		decl peeval = PeeValues(vcActual, m_mStats);
		if (m_bDblBoot)
		{
			if (peeval[0][0] >= 1) peeval ~= 1;
			else
				peeval ~= DoubleBoot(reverser(m_mStats),
							reverser(index'), peeval[0][0]);
		 }
		if (m_bFstDblBoot)
		{
			mstatsfd = sortr(mstatsfd);
			peeval ~= PeeValues(Quantiles((1 - peeval[][0])*m_cReps - 1,
													 mstatsfd), m_mStats); 
			pv = 2*peeval[][0] - PeeValues(vcActual, mstatsfd);
			peeval ~= (pv .> 1) .? 1 .: ((pv .< 0) .? 0 .: pv);
		}
		return peeval;
	}	
}		

Bootstrap::PrintHeading()
{
	if(m_bFstDblBoot && m_bDblBoot)
		PrintCall(0,"\n     Bootstrap p-values:",
					 "\n     Regular      Double       FD1          FD2");
	else
	if(m_bFstDblBoot)
		PrintCall(0,"\n     Bootstrap p-values:",
					 "\n     Regular      FD1          FD2");
	else
	if(m_bDblBoot)
		PrintCall(0,"\n     Bootstrap p-values:",
					 "\n     Regular      Double");
	else
		PrintCall(0,"\n     Bootstrap p-value: ");
}

Bootstrap::Settings(const iArgs0, ...)
{
	Defaults();
	if (iArgs0 == 1) m_bDblBoot = 1;
	else if (iArgs0 == 2) m_bFstDblBoot = 1;
	else if (iArgs0 == 3) m_bFstDblBoot = m_bDblBoot = 1;
	else if (iArgs0 == 4) m_bConfInt = 1; 
	decl args = va_arglist();
	if (sizeof(args) > 0 && isarray(args[0]))
		m_aNames = args[0];
	if (sizeof(args) > 1)
		if (args[1] > 0) m_cReps = args[1];
	if (sizeof(args) > 2 && isstring(args[2]))
	{
		m_OutFile = args[2];
		if(m_OutFile != "") m_bFile = 1;		
	}
	if (sizeof(args) > 3 && args[3] > 0) ranseed(args[3]);
	else ranseed(Currenttime());
	if (sizeof(args) > 4)
		if (args[4] > 0) m_cDblReps = args[4];
		else m_cDblReps = m_cReps;
	if (sizeof(args) > 5 && args[5] > 0) m_dAlphaMax = args[5];
	if (sizeof(args) > 6 && isstring(args[6]))
	{
		m_HFileIn = args[6];
		if(m_HFileIn != "") m_bCorrectedPval = 1;
	}
}

Bootstrap::TestCall(const mcData)
{
	decl output, starttime = time();
	decl actual = Estimate(mcData);
	if (rows(actual) > 1 && m_bDblBoot)
	{
		println("\nOnly one statistic allowed for the double bootstrap test.", 
					"\nRun terminating.");
		exit(0);					
	}
	while (rows(actual) > sizeof(m_aNames))
		m_aNames ~= "Statistic";	
	if (m_bConfInt)
	{
		decl critval = Test(LoadModel(mcData), actual);
		PrintCall(0,"\nActual values:"); 
		PrintCall(0, sprint("%r",m_aNames, actual));
		PrintCall(0, "\nQuantiles of the bootstrap distribution:");
		PrintCall(0,sprint("%r",m_aAlpha,"%c", m_aNames, critval'));		
		return actual~critval;		
		PrintCall(1,"Elapsed time: ", timespan(starttime));
	}
	else
	{
		decl peeval = Test(LoadModel(mcData), actual);
		if (m_bCorrectedPval)
			 decl corpval = PvalCorrect(peeval, loadmat(m_HFileIn));
		PrintCall(1,"");	 
		for (decl j = 0; j < rows(actual); j++)
		{
			PrintCall(0,"\n",m_aNames[j]," = ",actual[j]);
			PrintHeading();
			if (m_bCorrectedPval) PrintCall(0,"\nUncorrected ");
			PrintCall(0,peeval[j][]);
			if (m_bCorrectedPval)
			{
				PrintCall(0,"Size-corrected using ",m_HFileIn);
				PrintCall(0, corpval[j][]);
			}
		}	
		PrintCall(1,"\nElapsed time: ", timespan(starttime));
		return actual~peeval;
	}
}					   

/*-------------- end of Bootstrap class ---------------------------------*/

/*-----------------Simulation class (derived) --------------------------*/

class Simulation:Bootstrap
{
	decl m_mcResids;          
	decl m_mMCBins;
	decl m_cBins;   		                         
	decl m_cMCReps;                                        
	decl m_mMCdist;
	decl m_mMCordist;
	decl m_cWrtFreq;
	decl m_HFileOut;
	decl m_SizeCheckFile;
	decl m_mPvalfix;
	decl m_bError;
	decl m_cRepl;
	Simulation();
	virtual SetupReplication(const vrCase);
	SimDefaults();
	MCEdf(const vStats, const mMCDist);
	PrintLevels(const mMmcdist);
	PrintResults1(const mMCdist);
	PrintResults(const cA);
	SaveResults(const cRepl);
	Replication(const mPval);		
	Simulate(const vrCase, ...);
}

Simulation::Simulation()
{
	Bootstrap();
	SimDefaults();
}

Simulation::SimDefaults()
{
	m_cMCReps = 1000;
	m_cBins = m_cReps;
	m_cWrtFreq = 0;
	m_HFileOut = "sim_edf.mat";
	m_bError = 0;
}
Simulation::MCEdf(const vStats, const mMCDist)
{
	for (decl ac = 0; ac < rows(vStats); ac++)
	for (decl k = 0; k < columns(vStats); k++)          	
	for (decl i = 0; i < m_cBins; i++)
	{
		if (vStats[ac][k] > m_mMCBins[i]) ((mMCDist[0])[ac])[k][i]++;
		else break;			    
	}
}

Simulation::PrintLevels(const mMCdist)
{
	PrintCall(0,"\n1% test: ",(m_cRepl-(mMCdist[][ceil(.01*m_cBins)-1])')/(m_cRepl)); 
	PrintCall(0,"5% test: ",(m_cRepl-(mMCdist[][ceil(.05*m_cBins)-1])')/(m_cRepl)); 
	PrintCall(0,"10% test: ",(m_cRepl-(mMCdist[][ceil(.1*m_cBins)-1])')/(m_cRepl));
	PrintCall(0,"15% test: ",(m_cRepl-(mMCdist[][ceil(.15*m_cBins)-1])')/(m_cRepl));
}

Simulation::PrintResults1(const mMCdist)
{
	if(m_bFstDblBoot && m_bDblBoot)
		PrintLevels(mMCdist[0:3][]);
	else
	if(m_bFstDblBoot)
		PrintLevels(mMCdist[0:2][]);
	else
	if(m_bDblBoot)
		PrintLevels(mMCdist[0:1][]);
	else
		PrintLevels(mMCdist[0][]);
}

Simulation::PrintResults(const cA)
{
	PrintHeading();
	if (m_bCorrectedPval) PrintCall(0,"\nUncorrected:");
	PrintResults1(m_mMCdist[cA]);
	if (m_bCorrectedPval)
	{
		PrintCall(0,"Corrected using ",m_HFileIn,":");
		PrintResults1(m_mMCordist[cA]);
	}
}

Simulation::SaveResults(const cRepl)
{
	decl cstats = sizeof(m_mMCdist), outmat = <>;  
	for (decl j = 0; j<cstats; j++)
	outmat ~= (cRepl + 1 - m_mMCdist[j]')/(cRepl + 1);
	if (savemat(m_HFileOut, outmat))
		PrintCall(1,"Saved EDFs at iteration no. ",cRepl+1);
	else
		PrintCall(1,"EDFs not saved at iteration no. ",cRepl+1);
	if (m_bCorrectedPval)
	{
		outmat = <>;  
		for (decl j = 0; j<cstats; j++)
		outmat ~= (cRepl + 1 - m_mMCordist[j]')/(cRepl + 1);					
		if (savemat(m_SizeCheckFile, outmat))
			PrintCall(1,"Saved check-EDFs at iteration no. ",cRepl+1);
		else
			PrintCall(1,"check-EDFs not saved at iteration no. ",cRepl+1);
	}		
}

Simulation::Replication(const mPval)		
{
	decl check = m_bError;
	m_bError = 0;
	if (!check)
	{
		MCEdf(mPval, &m_mMCdist);
		if (m_bCorrectedPval)
			MCEdf(PvalCorrect(mPval, m_mPvalfix), &m_mMCordist);											
	}
	return check;	
}

Simulation::Simulate(const vrCase, ...)
{		
	decl args = va_arglist();
	SimDefaults();
	if (sizeof(args) > 0 && args[0] != 0) m_cMCReps = args[0];
	if (sizeof(args) > 1 && args[1] > 0) m_cWrtFreq = args[1];
	if (sizeof(args) > 2 && isstring(args[2]) && args[2] != "")
											m_HFileOut = args[2];
	m_SizeCheckFile = "c_"~m_HFileOut;
	decl repl, mcdata, cwrite = 0;
	m_cBins = m_cReps;
	m_mMCBins = (range(1,m_cBins)./(m_cBins)) - DBL_EPSILON;
	decl checkb = 0, errorcount = 0;
	decl cbootdists = 1 + m_bDblBoot + 2*m_bFstDblBoot;
	decl cstats = sizeof(Estimate(SetupReplication(vrCase)));
	m_mMCdist = new array[cstats];
	for (decl ac = 0; ac<cstats; ac++)
		m_mMCdist[ac] = zeros(cbootdists, m_cBins);
	if (m_bCorrectedPval)
	{
		m_mPvalfix = loadmat(m_HFileIn);
		m_mMCordist = m_mMCdist;
	}	
	decl mcresids, starttime = time();
	for (repl=0; repl<m_cMCReps; repl++)
	{
		cwrite++;
	
		mcdata = SetupReplication(vrCase);
		mcresids = LoadModel(mcdata);	 
		checkb = Replication(Test(mcresids, Estimate(mcdata)));								
		if (checkb)
		{
			PrintCall(1,"Estimation failure in replication ", repl);
			repl--;
			cwrite--;
			errorcount++;
			if (errorcount > m_cMCReps/10)
			{
				PrintCall (1,"Estimation failure in ", errorcount,
								" replications: run aborted");
				break;					
			}						
		}
		if (m_cWrtFreq) 
			if (cwrite >= m_cWrtFreq || repl >= m_cMCReps-1)
			{
				SaveResults(repl);
				cwrite = 0;
			}
	}
	m_cRepl = repl;
	for (decl ac = 0; ac < cstats; ac++)
	{
		PrintCall(1,"\nResults for ",m_aNames[ac]);
		PrintCall(1,"***   Estimated Powers   ***");
		PrintResults(ac);
	}	
	PrintCall(1,"\nElapsed time: ", timespan(starttime));
}

/*---------------------End of Simulation class ---------------------------*/ 


